"""
Basic Python Implementation example for Anybus-module Modbus-TCP
Data Type: 16-bit format A

Version: 202507-2

This document provides an example of how to implement the Modbus-TCP protocol in Python using the pyModbusTCP open-source library.
    Hardware Setup:
        Power Supply: A power supply compatible with the INT-MOD-ANY pluggable interface module.
        Interface Module: INT-MOD-ANY
        ModbusTCP Module: Anybus CompactCom M40 ModbusTCP (plugged into the INT-MOD-ANY module)
    Documentation:
        The implementation is based on the Fieldbus Implementation Guide, which is available on the website. 
        (https://www.delta-elektronika.nl/anybus-module)
    Purpose:
        This example demonstrates how to approach a Modbus-TCP implementation using the guide and the mentioned hardware. It is intended for educational or development purposes.
    Important Notes:
        No error handling is included in this code.
        Users should be cautious and understand the risks of using the code without proper validation and safety checks.

Disclaimer
This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
is not responsible for any damages, losses, or issues arising from its use,
implementation, or modification.

"""

# Required Libraries
from pyModbusTCP.client import ModbusClient
import struct
import time # Optional

# Configuration, IP-address of the Anybus module plugged into the INT-MOD-ANY
IPADDRESS_ANYBUSMODULE = "10.1.1.26"

# Power supply nominal output values for scaling
NOM_VOLTAGE_PSU = 0 # [V] # enter here the nominal output voltage of the PSU.
NOM_CURRENT_PSU = 0 # [A] # enter here the nominal output current of the PSU.


# --------------------------------------------------------------------------- #
# Numeric Literals and Data Types in Python (for Modbus use)
# --------------------------------------------------------------------------- #
# Python supports different numeric formats for register addresses and values:
#
# 1. Decimal (base 10):      Just write the number directly.
#      Example:  offset = 1024
#
# 2. Hexadecimal (base 16):  Prefix with '0x'. Common for Modbus register maps.
#      Example:  offset = 0x400  # same as 1024
#
# 3. Binary (base 2):        Prefix with '0b'. Used when working with bit flags.
#      Example:  control_word = 0b0000001000000001  # bits 0 and 9 set
#
# These are all treated as integers (type: int) in Python, so you can use them
# interchangeably in Modbus function calls like:
#   client.write_single_register(offset, value)
#
# --------------------------------------------------------------------------- #
# Floating-point values (REAL32) and Bitwise Representation
# --------------------------------------------------------------------------- #
# 4. 32-bit IEEE 754 Float:
#    When Modbus registers hold a float (REAL32), they are split across two 16-bit words.
#
#    To convert float → two registers:
#        low, high = struct.unpack(">HH", struct.pack(">f", float_value))
#
#    To convert two registers → float:
#        combined = (high << 16) | low
#        float_value = struct.unpack(">f", combined.to_bytes(4, 'big'))[0]
#
# This is used to write and read float values via:
#   client.write_multiple_registers(offset, [low, high])
#   client.read_holding_registers(offset, 2)


# --------------------------------------------------------------------------- #
# Methods used from pyModbusTCP for Read and Write
# --------------------------------------------------------------------------- #

# Modbus function READ_HOLDING_REGISTERS (0x03)
# read_holding_registers(reg_addr, reg_nb=1)
# Parameters
# reg_addr (int) – register address (0 to 65535)
# reg_nb (int) – number of registers to read (1 to 125)

# Returns
# registers list or None if fail

# Return type
# list of int or None


# Modbus function WRITE_MULTIPLE_REGISTERS (0x10)
# write_multiple_registers(regs_addr, regs_value)
#     Parameters
#     regs_addr (int) – registers address (0 to 65535)
#     regs_value (list) – registers values to write

#     Returns
#     True if write ok
#     Return type
#     bool

# --------------------------------------------------------------------------- #
# Register mapping from the implementation guide for Float Format
# --------------------------------------------------------------------------- #

register_map_Read_16bit = {
    # All readable param.
    'CVprg':(0x800,2), # (offset, word length)
    'CVmon':(0x801,2),
    'CCprg':(0x802,2),
    'CCmon':(0x803,2),
    'Refresh_Counter':(0x804,1),
    'Status_Register_A':(0x805,1),
}

register_map_Write_16bit = {
    # All writable param.
    'CVprg':(0x000,2), 
    'CCprg':(0x001,2), 
    'RemCTRL':(0x002,1), 
}

# --------------------------------------------------------------------------- #
# General Helper functions
# --------------------------------------------------------------------------- #

def connect(address):
    """
    Open connection to the Anybus module
    """
    client = ModbusClient(address, port=502, unit_id=1, auto_open=True)
    client.open()
    return client


def disconnect(client):
    """
    Close the connection to the Anybus module
    """
    client.close()

# --------------------------------------------------------------------------- #
# 16-Bit format A, Helper functions
# --------------------------------------------------------------------------- #

def scale_cv(value, operation):
    """
    Scaling of CVprg and CVmon to nominal parameter value
    """
    nominal = 62500 # nominal value of voltage parameters
    if operation == 'write':
        # Value is desired value to be programmed
        scaled = (nominal/NOM_VOLTAGE_PSU)*value
        return scaled
    elif operation == 'read':
        # Value is actual monitoring value at the output
        scaled = (NOM_VOLTAGE_PSU/nominal)*value
        return scaled
    

def scale_cc(value, operation):
    """
    Scaling of CCprg and CCmon to nominal parameter value
    """
    nominal = 31250 # nominal value of current parameters

    if operation == 'write':
        # Value is desired value to be programmed
        scaled = (nominal/NOM_CURRENT_PSU)*value
        return scaled

    elif operation == 'read':
        # Value is actual monitoring value at the output
        scaled = (NOM_CURRENT_PSU/nominal)*value
        return scaled


def read_param_16bit_format(client, parameter):
    """
    Read from Anybus module function for 16-bit format
    """
    # Unpack tuple from the register map
    offset, words = register_map_Read_16bit[parameter]
    data_rec = client.read_holding_registers(offset, words)  # FC 0x03
    # Parameter of single word length
    if parameter in ('CVprg', 'CVmon'):
        return round(scale_cv(data_rec[0], 'read'),2) # Apply scaling to convert to actual value.

    elif parameter in ('CCprg', 'CCmon'):
        return round(scale_cc(data_rec[0], 'read'),2) # Apply scaling to convert to actual value.
    else:
        return int(data_rec[0])


def write_param_16bit_form(client, parameter, value):
    """
    Write to Anybus module function, 16-bit format
    """
    # Unpack tuple from the register map
    offset, words = register_map_Write_16bit[parameter]

    if parameter == 'CVprg':
        value = scale_cv(value, 'write')
    elif parameter == 'CCprg':
        value = scale_cc(value, 'write')
    else:
        value = value

    client.write_single_register(offset, int(value))

# --------------------------------------------------------------------------- #
# Additional Helper Functions
# --------------------------------------------------------------------------- #

def decode_status_register_a(value):
    """
    Decode the binary value representing Status Register A
    """
    value = int(value)
    bits = {
        0: "CV", 1: "CC", 2: "CP", 3: "Vlim", 4: "Ilim", 5: "Plim",
        6: "DcfVolt", 7: "DcfCurr", 8: "OT", 9: "PSOL", 10: "ACF",
        11: "Interlock", 12: "RSD", 13: "Output", 14: "FrontpanelLock"
    }
    return {name: bool(value & (1 << bit)) for bit, name in bits.items()}


def encode_remctrl(rsd, output, rem_cv, rem_cc):
    """
    Return the 16-bit word for the RemCTRL register (flags must be 0 or 1).
    """
    if not all(flag in (0, 1) for flag in (rsd, output, rem_cv, rem_cc)):
        raise ValueError("All flags must be 0 or 1")
    return rsd | (output << 1) | (rem_cv << 9) | (rem_cc << 10)



def read_print_all(client):
    """
    Print all parameters in pretty print formatting
    """
    # Read param
    cv_prg = read_param_16bit_format(client, "CVprg")   # V
    cc_prg = read_param_16bit_format(client, "CCprg")   # A
    cv_mon = read_param_16bit_format(client, "CVmon")   # V
    cc_mon = read_param_16bit_format(client, "CCmon")   # A
    refresh = read_param_16bit_format(client, "Refresh_Counter")
    raw_a = read_param_16bit_format(client, "Status_Register_A")

    # Print
    print("\n====================================")
    print(f"CVprg:                  {cv_prg:.2f} V")
    print(f"CCprg:                  {cc_prg:.2f} A\n")
    print(f"CVmon:                  {cv_mon:.2f} V")
    print(f"CCmon:                  {cc_mon:.2f} A\n")
    print(f"Status_Register_A:      {raw_a:016b}")
    print(f"Refresh Counter:        {refresh}")
    print("====================================\n")


def read_print_register_a(client):
    """
    Pretty print of status registers A
    """
    # Read and decode
    decoded_a = decode_status_register_a(
        read_param_16bit_format(client, "Status_Register_A"))

    # Print vertically
    print("\n====================================")
    print("Status Register A")
    for name, state in decoded_a.items():
        print(f"  {name:<18}: {'ON' if state else 'off'}")

# --------------------------------------------------------------------------- #
# Demonstration Examples
# --------------------------------------------------------------------------- #

# Steps before connecting:
# 1. Configure all hardware as described in the product manuals.
# 2. Obtain the IP-address of the Anybus module through the web interface of the power supply.
# 3. Set the INT-MOD-ANY data format to Float format in the web interface of the power supply.
# 4. Safety first, dont connect any hardware to the OUTPUT of the power supply. Its at own risk.

# The demonstration examples show how programming and monitoring can be achieved.
# Functions are basic, it is assumed that when the reader understands the basic operation,
# that it is easy to customize or to add error handling or to wrap the code into a class.

# Open connection with the Anybus module, should be closed later on.
Anybus_connection = connect(IPADDRESS_ANYBUSMODULE)

# Note: To read parameters from the PSU, the programming source of the power supply,
# must be set to the INT-MOD-ANY. This can be done in multiple ways.
# By selecting it in the PSU's web server or sending remCV, remCC, remCP commands.

# Note: Time-delays are added optionally.

# Warning! Change the nominal voltage and current of the psu at the top of this file!
# Otherwise set voltage and current will be scaled wrong in both programming and monitoring.

# Example 1: Read a parameter of continuous type.
data = read_param_16bit_format(Anybus_connection, 'CVprg')
print(data)
data = read_param_16bit_format(Anybus_connection, 'CCmon')
print(data)

# Example 2: Read a parameter of register type.
data = read_param_16bit_format(Anybus_connection, 'Status_Register_A')
print(data)

# Additional: Decode the previously read Status Register A
decoded_data = decode_status_register_a(data)
print(decoded_data)
print(decoded_data["CV"])  # extract single value.

# ! Change nominal voltage and current of the power supply at top of this file first.
# Example 3: Write a parameter of continuous type.
write_param_16bit_form(Anybus_connection, 'CVprg', 0.0) # Replace zero with desired voltage [V]
time.sleep(0.1)

# Example 4: Write a parameter of register type.
write_param_16bit_form(Anybus_connection, 'RemCTRL', 2) # Sets OUTPUT to 1 and all others to 0.
time.sleep(0.1)

# Example 5: Write a parameter of register type, but set multiple settings at once.
RSD = 0
OUTPUT = 1
REM_CV = 1
REM_CC = 1
write_param_16bit_form(Anybus_connection, 'RemCTRL',
            encode_remctrl(RSD, OUTPUT, REM_CV, REM_CC))
time.sleep(0.1)

# Additional Example: Print all readable parameters
read_print_all(Anybus_connection)
time.sleep(0.1)

# Additional Example: Read and print registers in human friendly format
read_print_register_a(Anybus_connection)
time.sleep(0.1)

# Close the connection
disconnect(Anybus_connection)
